#!/usr/bin/env python
"""
Aggregate per‑condition summaries into a single CSV for geometry baseline.

This script reads the ``summary.csv`` generated by ``run_geometry.py``
from the specified runs directory and copies it into a results directory.
If the summary file does not exist, it scans all JSON files under
``runs/`` and constructs a summary table on the fly.
"""

import os
import json
import argparse
import pandas as pd

def main():
    parser = argparse.ArgumentParser(description="Aggregate geometry baseline results.")
    parser.add_argument("--runs-dir", type=str, default="runs", help="Directory containing run outputs.")
    parser.add_argument("--out-dir", type=str, default="results", help="Directory to write aggregated results.")
    args = parser.parse_args()
    runs_dir = args.runs_dir
    out_dir = args.out_dir
    os.makedirs(out_dir, exist_ok=True)
    summary_csv = os.path.join(runs_dir, "summary.csv")
    if os.path.exists(summary_csv):
        df = pd.read_csv(summary_csv)
    else:
        # Reconstruct summary from JSON files.
        rows = []
        for fname in os.listdir(os.path.join(runs_dir, "runs")):
            if not fname.endswith(".json"):
                continue
            with open(os.path.join(runs_dir, "runs", fname), "r", encoding="utf-8") as f:
                summ = json.load(f)
            row = {
                "gauge": summ["gauge"],
                "L": summ["L"],
                "b": summ["b"],
                "kappa": summ["kappa"],
                "f": summ["f"],
                "seed": summ["seed"],
                "ell": summ["ell"],
                "s_phi": summ["phi_slopes"]["s_phi"],
                "r2_phi": summ["phi_slopes"]["r2_phi"],
                "s_grad": summ["grad_slopes"]["s_grad"],
                "r2_grad": summ["grad_slopes"]["r2_grad"],
                "s_grad_norm": summ["grad_norm_slopes"]["s_grad_norm"],
                "r2_grad_norm": summ["grad_norm_slopes"]["r2_grad_norm"],
            }
            for lens in summ["lensing"]:
                lam = lens["lambda"]
                row[f"alpha_slope_lambda{lam}"] = lens["slope"]
                row[f"alpha_r2_lambda{lam}"] = lens["r2"]
            rows.append(row)
        df = pd.DataFrame(rows)
    # Save aggregated summary.
    out_path = os.path.join(out_dir, "summary.csv")
    df.to_csv(out_path, index=False)
    print(f"Aggregated summary written to {out_path}")

if __name__ == "__main__":
    main()